/*
  LCAP:  Linux Kernel Capability Remover
  Copyright (C) 1999  spoon@ix.netcom.com
  Copyright (C) 2000  spoon@ix.netcom.com
  
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
*/

/* $Id: lcap.c,v 1.10 1999/12/22 15:26:05 spoon Exp $ */

#include <stdio.h>
#include <linux/capability.h>
#include <errno.h>
#include <string.h>
#include <getopt.h>
#include <stdlib.h>
#include <ctype.h>

#define PROC_CAP "/proc/sys/kernel/cap-bound"

typedef __u32 kernel_cap_t;

#define CAP_FULL_SET (~0)
#define CAP_TO_MASK(flag) (1 << (flag))
#define cap_raise(cap, flag)  (cap |= CAP_TO_MASK(flag))
#define cap_lower(cap, flag)  (cap &= ~CAP_TO_MASK(flag))
#define cap_raised(cap, flag) (cap & CAP_TO_MASK(flag) & CAP_FULL_SET)

  static char *thecaps[] =
    {
    "CAP_CHOWN",            "chown(2)/chgrp(2)",
    "CAP_DAC_OVERRIDE",     "DAC access",
    "CAP_DAC_READ_SEARCH",  "DAC read",
    "CAP_FOWNER",           "owner ID not equal user ID",
    "CAP_FSETID",           "effective user ID not equal owner ID",
    "CAP_KILL",             "real/effective ID not equal process ID",
    "CAP_SETGID",           "setgid(2)",
    "CAP_SETUID",           "set*uid(2)",
    "CAP_SETPCAP",          "transfer capability",
    "CAP_LINUX_IMMUTABLE",  "immutable and append file attributes",
    "CAP_NET_BIND_SERVICE", "binding to ports below 1024",
    "CAP_NET_BROADCAST",    "broadcasting/listening to multicast",
    "CAP_NET_ADMIN",        "interface/firewall/routing changes",
    "CAP_NET_RAW",          "raw sockets",
    "CAP_IPC_LOCK",         "locking of shared memory segments",
    "CAP_IPC_OWNER",        "IPC ownership checks",
    "CAP_SYS_MODULE",       "insertion and removal of kernel modules",
    "CAP_SYS_RAWIO",        "ioperm(2)/iopl(2) access",
    "CAP_SYS_CHROOT",       "chroot(2)",
    "CAP_SYS_PTRACE",       "ptrace(2)",
    "CAP_SYS_PACCT",        "configuration of process accounting",
    "CAP_SYS_ADMIN",        "tons of admin stuff",
    "CAP_SYS_BOOT",         "reboot(2)",
    "CAP_SYS_NICE",         "nice(2)",
    "CAP_SYS_RESOURCE",     "setting resource limits",
    "CAP_SYS_TIME",         "setting system time",
    "CAP_SYS_TTY_CONFIG",   "tty configuration",
    NULL, NULL,
    };

  static void usage(const char *argv0)
    {
    fprintf(stderr, "Usage: %s [-h]\n", argv0);
    fprintf(stderr, "       -h = you're looking at it\n");
    fprintf(stderr, "Usage: %s [-v[v]] -c cap\n", argv0);
    fprintf(stderr, "       -v = verbose mode\n");
    fprintf(stderr, "       -c = return 1 if capability is set, 0 if not\n");
    fprintf(stderr, "       cap = capability to check\n");
    fprintf(stderr, "Usage: %s [-v[v]] [-z] cap ...\n", argv0);
    fprintf(stderr, "       -v = verbose mode\n");
    fprintf(stderr, "       -z = zero out capability set\n");
    fprintf(stderr, "       cap = with -z, capabilities to not zero out\n");
    fprintf(stderr, "       cap = without -z, capabilities to zero out\n");
    };

  static void listcaps(const kernel_cap_t caps)
    {
    int index=0;

    printf("Current capabilities: 0x%08X\n", caps);
    while (thecaps[index*2] != NULL)
      {
      printf("  %2d) %c%-25s", index, cap_raised(caps, index) ? '*' : ' ',
             thecaps[index*2]);
      if ((index) % 2)
        printf("\n");
      index++;
      };
    printf("\n");
    printf("    * = Capabilities currently allowed\n");
    };

int main(int argc, char *argv[])
  {
  kernel_cap_t caps=0;
  FILE *fptr;
  int index;
  char letter;
  short checkflag=0, errorflag=0, verboseflag=0, zeroflag=0;

  /* open the /proc file */
  if ((fptr=fopen(PROC_CAP, "r")) == NULL)
    {
    perror(PROC_CAP);
    exit(errno);
    };
  /* snag the current setting */
  fscanf(fptr, "%d", &caps);
  fclose(fptr);

  while ((letter=getopt(argc, argv, "chvz")) != -1)
    {
    switch (letter)
      {
      case 'c':
        checkflag = 1;
        break;
      case 'v':
        verboseflag++;
        break;
      case 'z':
        zeroflag = 1;
        break;
      case 'h':
      default:
        errorflag = 1;
        break;
      };
    };

  if (((checkflag) && (optind+1 != argc)) ||
      ((argc == optind) && (argc > 1)) ||
      (errorflag))
    {
    usage(argv[0]);
    exit(1);
    };

  if (argc == 1)
    {
    listcaps(caps);
    exit(0);
    };

  if (verboseflag >= 1)
    {
    printf("Current capabilities: 0x%08X\n", caps);
    };

  if (checkflag)
    {
    int capnum=-1;
    char *capstr=NULL, *capdescstr=NULL;

    if (isdigit(argv[optind][0]))
      {
      capnum = (int)strtol(argv[optind], (char **)NULL, 10);
      capstr = thecaps[capnum*2];
      capdescstr = thecaps[capnum*2+1];
      }
    else
      {
      int jndex=0;

      while (thecaps[jndex] != NULL)
        {
        if (!strcmp(argv[optind], thecaps[jndex]))
          {
          capnum = jndex/2;
          capstr = thecaps[jndex];
          capdescstr = thecaps[jndex+1];
          };
        jndex += 2;
        };
      };

    if (capnum == -1)
      {
      fprintf(stderr, "    %s: invalid capability\n", argv[2]);
      exit(1);
      };

    if (verboseflag >= 1)
      {
      printf("  %2d) %c%-25s\n", capnum, cap_raised(caps, capnum) ? '*' : ' ',
             thecaps[capnum*2]);
      printf("    * = Capability currently allowed\n");
      };

    exit(cap_raised(caps,capnum) ? 1 : 0);
    };

  if (verboseflag >= 1)
    {
    if (zeroflag)
      printf("  Setting");
    else
      printf("  Removing");
    printf(" capabilities:\n");
    };

  if (zeroflag)
    {
    caps = 0;
    };

  for (index=optind; index < argc; index++)
    {
    int capnum=-1;
    char *capstr=NULL, *capdescstr=NULL;

    if (isdigit(argv[index][0]))
      {
      capnum = (int)strtol(argv[index], (char **)NULL, 10);
      capstr = thecaps[capnum*2];
      capdescstr = thecaps[capnum*2+1];
      }
    else
      {
      int jndex=0;

      while (thecaps[jndex] != NULL)
        {
        if (!strcmp(argv[index], thecaps[jndex]))
          {
          capnum = jndex/2;
          capstr = thecaps[jndex];
          capdescstr = thecaps[jndex+1];
          };
        jndex += 2;
        };
      };

    if (capnum == -1)
      {
      fprintf(stderr, "    %s: invalid capability\n", argv[index]);
      exit(1);
      };

    if (verboseflag >= 1)
      printf("    %2d) %-22s %s\n", capnum, capstr, capdescstr);

    if (zeroflag)
      cap_raise(caps, capnum);
    else
      cap_lower(caps, capnum);
    };

  if (verboseflag >= 2)
    {
    listcaps(caps);
    };

  /* open the /proc file */
  if ((fptr=fopen(PROC_CAP, "w")) == NULL)
    {
    perror(PROC_CAP);
    exit(errno);
    };
  /* write the new setting */
  fprintf(fptr, "0x%x", caps);
  fclose(fptr);

  return(0);
  };

